﻿using System;
using System.Linq;
using System.Text;
using System.Globalization;

namespace Framework
{
    /// <summary>
    /// string の拡張メソッドです。
    /// </summary>
    public static partial class StringExtension
    {
        /// <summary>
        /// 文字列が空、もしくはnullであるかどうかを判定します。
        /// </summary>
        public static bool IsEmpty(this string target)
        {
            return string.IsNullOrEmpty(target);
        }

        /// <summary>
        /// 文字列が空でもnullでもないかどうかを判定します。
        /// </summary>
        public static bool IsNotEmpty(this string target)
        {
            return string.IsNullOrEmpty(target) == false;
        }

        /// <summary>
        /// 文字列が空、もしくはnull、もしくは空白のみかどうかを判定します。
        /// </summary>
        public static bool IsBlank(this string target)
        {
            return (target ?? "").Trim().Length == 0;
        }

        /// <summary>
        /// 文字列が空でもnullでも空白のみでもないかどうかを判定します。
        /// </summary>
        public static bool IsNotBlank(this string target)
        {
            return (target ?? "").Trim().Length != 0;
        }

        /// <summary>
        /// カンマにより文字列を分割します。
        /// カンマが連続した場合、空の文字列が結果の配列に含まれます。
        /// </summary>
        /// <returns>カンマで分割した文字列の配列。
        /// 分割対象がnullの場合はnull.</returns>
        public static string[] SplitByConma(this string target)
        {
            if (target == null)
            {
                return null;
            }

            return target.Split(',');
        }

        /// <summary>
        /// 指定した文字により文字列を分割します。
        /// このメソッドの動作はString.Split(char[])に準じます。
        /// </summary>
        /// <returns>separatorにより分割した文字列の配列。
        /// 分割対象がnullの場合はnull.</returns>
        /// <exception cref="ArgException">separatorにnullを指定した場合</exception>
        public static string[] SplitBy(this string target, params char[] separator)
        {
            if (target == null)
            {
                return null;
            }

            return target.Split(separator);
        }

        /// <summary>
        /// 指定した文字列により文字列を分割します。
        /// このメソッドの動作はString.Split(string[], StringSplitOption)の第二引数にNoneを渡したものに準じます。
        /// "hoge, piyo,foo".SplitBy(",", ", ") と、
        /// "hoge, piyo,foo".SplitBy(", ", ",") の結果が異なることに注意してください。
        /// </summary>
        /// <returns>separatorにより分割した文字列の配列。
        /// 分割対象がnullの場合はnull.</returns>
        /// <exception cref="ArgException">separatorにnullを指定した場合</exception>
        public static string[] SplitBy(this string target, params string[] separator)
        {
            if (target == null)
            {
                return null;
            }

            return target.Split(separator, StringSplitOptions.None);
        }

        /// <summary>
        /// string.Format(target, arg0) を返します。
        /// </summary>
        public static string Fmt(this string target, object arg0)
        {
            return string.Format(target, arg0);
        }

        /// <summary>
        /// string.Format(target, args) を返します。
        /// </summary>
        public static string Fmt(this string target, params object[] args)
        {
            return string.Format(target, args);
        }

        /// <summary>
        /// 対象文字列がnullまたは空文字列の場合、defaultStrを返します。
        /// そうでない場合、対象文字列を返します。
        /// </summary>
        public static string DefaultIfEmpty(this string target, string defaultStr)
        {
            return string.IsNullOrEmpty(target) ? defaultStr : target;
        }

        /// <summary>
        /// 文字列の最初の文字を取得します。
        /// nullで起動した場合、例外が発生します。
        /// </summary>
        public static char First(this string target)
        {
            if (target == null)
                throw ArgException.UnexpectedNull().butArgIsNull("target");
            return target[0];
        }

        /// <summary>
        /// 文字列の最後の文字を取得します。
        /// nullで起動した場合、例外が発生します。
        /// </summary>
        public static char Last(this string target)
        {
            if (target == null)
                throw ArgException.UnexpectedNull().butArgIsNull("target");
            return target[target.Length - 1];
        }
        /// <summary>
        /// 文字列target を T型(int, doubleなど)に変換する汎用メソッドです。
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="target"></param>
        /// <param name="f"></param>
        /// <returns></returns>
        static T ToT<T>(this string target, Func<string, Tuple<bool, T>> f) where T : struct
        {
            if (target == null)
                throw ArgException.UnexpectedNull().butArgIsNull("target");

            Tuple<bool, T> result = f(target);
            if (result._1)
                return result._2;

            throw ArgException.ExpectedFormatIs("10 digit number").but("target", target);
        }

        /// <summary>
        /// 文字列target を T型(int, doubleなど)に変換する汎用メソッドです。
        /// </summary>
        /// <typeparam name="T"></typeparam>
        /// <param name="target"></param>
        /// <param name="r"></param>
        /// <param name="f"></param>
        /// <returns></returns>
        static T ToT<T>(this string target, Radix r, Func<string, int, T> f) where T : struct
        {
            if (target == null)
                throw ArgException.UnexpectedNull().butArgIsNull("target");

            try
            {
                return f(target, (int)r);
            }
            catch (FormatException)
            {
                throw ArgException.ExpectedFormatIs(r.ToFormat()).but("target", target);
            }
        }

        /// <summary>
        /// 文字列をintに変換します。
        /// nullを変換しようとした場合や、文字列が10進表現でなかった場合、
        /// intであらわせる範囲を超えていた場合は例外が発生します。
        /// </summary>
        public static int ToInt(this string target)
        {
            return target.ToT(str =>
            {
                int result;
                if (int.TryParse(str, out result) == true)
                {
                    return Tuple.Create(true, result);
                }

                return Tuple.Create(false, 0);
            });
        }

        /// <summary>
        /// 文字列を指定基数でintに変換します。
        /// nullを変換しようとした場合や、文字列が指定基数表現でなかった場合、
        /// intであらわせる範囲を超えていた場合は例外が発生します。
        /// また、指定基数がRadix.Deci以外の場合、負の数は指定できません。
        /// </summary>
        public static int ToInt(this string target, Radix radix)
        {
            return target.ToT(radix, (str, r) => Convert.ToInt32(str, r));
        }

        /// <summary>
        /// 文字列をintに変換します。
        /// nullを変換しようとした場合や、文字列が10進表現でなかった場合は
        /// 指定の規定値を返します。
        /// </summary>
        /// <param name="target"></param>
        /// <param name="defaultValue"></param>
        /// <returns></returns>
        public static int ToInt(this string target, int defaultValue)
        {
            int result;
            if (!int.TryParse(target, out result))
            {
                result = defaultValue;
            }

            return result;
        }

        /// <summary>
        /// 文字列をlongに変換します。
        /// nullを変換しようとした場合や、文字列が10進表現でなかった場合、
        /// longであらわせる範囲を超えていた場合は例外が発生します。
        /// </summary>
        public static long ToLong(this string target)
        {
            return target.ToT(str =>
            {
                long result;
                if (long.TryParse(str, out result))
                    return Tuple.Create(true, result);
                return Tuple.Create(false, 0L);
            });
        }

        /// <summary>
        /// 文字列を指定基数でlongに変換します。
        /// nullを変換しようとした場合や、文字列が指定基数表現でなかった場合、
        /// longであらわせる範囲を超えていた場合は例外が発生します。
        /// また、指定基数がRadix.Deci以外の場合、負の数は指定できません。
        /// </summary>
        public static long ToLong(this string target, Radix radix)
        {
            return target.ToT(radix, (str, r) => Convert.ToInt64(str, r));
        }

        /// <summary>
        /// 文字列をbyteに変換します。
        /// nullを変換しようとした場合や、文字列が10進表現でなかった場合、
        /// byteであらわせる範囲を超えていた場合は例外が発生します。
        /// </summary>
        public static byte ToByte(this string target)
        {
            return target.ToT(str =>
            {
                byte result;
                if (byte.TryParse(str, out result) == true)
                {
                    return Tuple.Create(true, result);
                }

                return Tuple.Create(false, (byte)0);
            });
        }

        /// <summary>
        /// 文字列をdoubleに変換します。
        /// nullを変換しようとした場合や、文字列が10進表現でなかった場合、
        /// doubleであらわせる範囲を超えていた場合は例外が発生します。
        /// </summary>
        public static double ToDouble(this string target)
        {
            return target.ToT(str =>
            {
                double result;
                if (double.TryParse(str, out result) == true)
                {
                    return Tuple.Create(true, result);
                }

                return Tuple.Create(false, .0);
            });
        }

        /// <summary>
        /// 文字列をdecimalに変換します。
        /// nullを変換しようとした場合や、文字列が10進表現でなかった場合、
        /// decimalであらわせる範囲を超えていた場合は例外が発生します。
        /// </summary>
        public static decimal ToDecimal(this string target)
        {
            return target.ToT(str =>
            {
                decimal result;
                if (decimal.TryParse(str, out result) == true)
                {
                    return Tuple.Create(true, result);
                }

                return Tuple.Create(false, 0m);
            });
        }

        /// <summary>
        /// 文字列をDateTimeオブジェクトに変換します。
        /// nullを変換しようとした場合や、文字列が10進表現でなかった場合、
        /// DateTimeの書式にのっとっていなかった場合は例外が発生します。
        /// </summary>
        public static DateTime ToDateTime(this string target)
        {
            if (target == null)
            {
                throw ArgException.UnexpectedNull().butArgIsNull("target");
            }

            DateTime result;
            if (DateTime.TryParse(target, out result) == true)
            {
                return result;
            }

            throw ArgException.ExpectedFormatIs("datetime").but("target", target);
        }

        /// <summary>
        /// 文字列を指定書式でDateTimeオブジェクトに変換します。
        /// nullを変換しようとした場合や、文字列が10進表現でなかった場合、書式が指定されなかった場合、
        /// DateTimeの書式にのっとっていなかった場合は例外が発生します。
        /// </summary>
        public static DateTime ToDateTime(this string target, params string[] formats)
        {
            if (target == null)
            {
                throw ArgException.UnexpectedNull().butArgIsNull("target");
            }

            if (formats == null)
            {
                throw ArgException.UnexpectedNull().butArgIsNull("formats");
            }

            if (formats.Length == 0)
            {
                throw ArgException.ExpectedGreaterThan(0).but("formats.Length", formats.Length);
            }

            DateTime result;
            if (DateTime.TryParseExact(target, formats, DateTimeFormatInfo.CurrentInfo, DateTimeStyles.None, out result) == true)
            {
                return result;
            }

            throw ArgException.ExpectedFormatIs("datetime").but("target", target);
        }

        /// <summary>
        /// IndexOfの"愛々"問題対応
        /// </summary>
        public static int IdxOf(this string target, string str)
        {
            return target.IndexOf(str, StringComparison.Ordinal);
        }

        /// <summary>
        ///  LastIndexOfの"愛々"問題対応
        /// </summary>
        public static int LastIdxOf(this string target, string str)
        {
            return target.LastIndexOf(str, StringComparison.Ordinal);
        }
        
        /// <summary>
        /// 文字列の先頭から指定された長さだけ取得します。
        /// nullで起動した場合、nullが返ります。
        /// lenに0以下の数値を指定した場合、空文字列が返ります。
        /// </summary>
        public static string Head(this string target, int len)
        {
            if (target == null)
            {
                return null;
            }
            if (len <= 0)
            {
                return "";
            }

            var targetLen = target.Length;
            if (targetLen < len)
            {
                len = targetLen;
            }

            return target.Substring(0, len);
        }

        /// <summary>
        /// 文字列の末尾から指定された長さだけ取得します。
        /// nullで起動した場合、nullが返ります。
        /// lenに0以下の数値を指定した場合、空文字列が返ります。
        /// </summary>
        public static string Tail(this string target, int len)
        {
            if (target == null)
            {
                return null;
            }
            if (len <= 0)
            {
                return "";
            }

            var targetLen = target.Length;
            if (targetLen < len)
            {
                len = targetLen;
            }

            return target.Substring(targetLen - len, len);
        }

        /// <summary>
        /// 文字列の先頭から指定された長さを削った文字列を取得します。
        /// nullで起動した場合、nullが返ります。
        /// lenに0以下の数値を指定した場合、文字列がそのまま返ります。
        /// </summary>
        public static string Drop(this string target, int len)
        {
            if (target == null)
                return null;

            var targetLen = target.Length;
            if (targetLen < len)
                return "";
            if (len <= 0)
                return target;
            return target.Substring(len);
        }

        /// <summary>
        /// 文字列の末尾から指定された長さを削った文字列を取得します。
        /// nullで起動した場合、nullが返ります。
        /// lenに0以下の数値を指定した場合、文字列がそのまま返ります。
        /// </summary>
        public static string DropLast(this string target, int len)
        {
            if (target == null)
                return null;

            var targetLen = target.Length;
            if (targetLen < len)
                return "";
            if (len <= 0)
                return target;
            return target.Substring(0, targetLen - len);
        }

        /// <summary>
        /// 文字列の末尾に最も近いセパレータ以前の文字列を返します。
        /// セパレータが文字列中に存在しない場合、文字列がそのまま返ります。
        /// nullで起動した場合、nullが起動します。
        /// セパレータがnullの場合、文字列がそのまま返りますが、空文字列の場合は空文字列が返ることに注意してください。
        /// </summary>
        public static string SubstrBeforeLast(this string target, string separator)
        {
            if (target == null)
                return null;
            if (separator == null)
                return target;
            if (separator.Length == 0)
                return "";

            var pos = target.LastIdxOf(separator);
            return pos == -1 ? target
                             : target.Substring(0, pos);
        }

        /// <summary>
        /// セパレータ以後の文字列を返します。
        /// セパレータが文字列中に存在しない場合、空文字列が返ります。
        /// nullで起動した場合、nullが起動します。
        /// セパレータがnullの場合、空文字列が返りますが、空文字列の場合は文字列がそのまま返ることに注意してください。
        /// </summary>
        public static string SubstrAfter(this string target, string separator)
        {
            if (target == null)
            {
                return null;
            }
            if (separator == null)
            {
                return "";
            }

            var len = separator.Length;
            if (len == 0)
            {
                return target;
            }

            var pos = target.IdxOf(separator);
            return pos == -1 ? ""
                             : target.Substring(pos + len);
        }

        /// <summary>
        /// 文字列を改行文字で分割し、各行の先頭の空白を除去した後、さらに先頭の一文字を除去して連結した文字列を返します。
        /// SQL等を逐次文字列で表す場合などに、このメソッドを使用することでより自然に表すことができます。
        /// 最終的に返される文字列の改行文字は、System.Console.Out.NewLineで返される改行文字に統一されます。
        /// </summary>
        public static string StripMargin(this string target)
        {
            if (target == null)
                return null;
            if (target.Length < 2)
                return target;

            var result = new StringBuilder(target.Length);
            var newLine = Console.Out.NewLine;
            foreach (var line in target.SplitBy("\r\n", "\r", "\n"))
            {
                result.Append(line.TrimStart(' ').Drop(1)).Append(newLine);
            }
            var newLineLen = newLine.Length;
            result.Remove(result.Length - newLineLen, newLineLen);
            return result.ToString();
        }

    }
}
